/**
 * 
 */
package gov.va.med.mhv.vamf.util;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;


/**
 * @author DNS
 *
 */
public class VamfJwtUtility {
	
	private static final Log LOG = LogFactory.getLog(VamfJwtUtility.class);
	
	private static final String DF_JWT_DOB = "yyyy-MM-dd";
	
	private Algorithm signingAlgorithm;
	
	private String issuerId;
	
	private Integer notBeforeMinutes;
	
	private Integer expiresAfterMinutes;
	
	public VamfJwtUtility(String issuerId, Integer notBeforeMinutes, Integer expireAfterMinutes, String privateKey) {
		this.issuerId = issuerId;
		this.notBeforeMinutes = notBeforeMinutes;
		this.expiresAfterMinutes = expireAfterMinutes;
		this.signingAlgorithm = Algorithm.RSA512(JwtSigner.loadPrivateKey(privateKey));
	}
	
	public String createMobileWebToken(String firstName, String middleName, String lastName, String icn, String ssn, String email, String gender, Date birthDate, String mhvId, String authenticationAuthority, String mhvSessionId) {
		
		String jwtGender = "";
		if(gender != null) {
			if("Female".equalsIgnoreCase(gender)) {
				jwtGender = "F";
			}
			else if("Male".equalsIgnoreCase(gender)) {
				jwtGender = "M";
			}
		}
		
		String jwtAuthenticationAuthority = "";
		if(authenticationAuthority != null) {
			if(authenticationAuthority.toUpperCase().contains("DSLOGON")) {
				jwtAuthenticationAuthority = "DSLOGON";
			}
			else {
				jwtAuthenticationAuthority = "MHV";
			}
		}
		
		if(LOG.isDebugEnabled()) {
			LOG.debug(String.format("Creating mobile web token with paramters First Name: %s, Last Name: %s, ICN: %s, Email: %s, MHV ID: %s.", 
					firstName, lastName, icn, email, mhvId));
		}
		
		// setup some dates used when creating the web token
		Calendar expireAfter = Calendar.getInstance();
		expireAfter.add(Calendar.MINUTE, expiresAfterMinutes);
		
		Calendar notBefore = Calendar.getInstance();
		notBefore.add(Calendar.MINUTE, (-1 * notBeforeMinutes));
		
		
		String dob = (birthDate != null) ? new SimpleDateFormat(DF_JWT_DOB).format(birthDate) : "";
		
		if(LOG.isDebugEnabled()) {
			LOG.debug(String.format("Current Time: %s, Not Before: %s, JWT Expire After: %s", new Date(), notBefore.getTime(), expireAfter.getTime()));
		}
		
		// create the input JSON web token
		String token = JWT.create()
				.withClaim(VamfJwtClaimsConstants.AUTHENTICATED, Boolean.TRUE)
                .withClaim(VamfJwtClaimsConstants.USER_FIRST_NAME, firstName)
                .withClaim(VamfJwtClaimsConstants.USER_LAST_NAME, lastName)
                .withClaim(VamfJwtClaimsConstants.USER_MIDDLE_NAME, (middleName==null)?"":middleName)
                .withClaim(VamfJwtClaimsConstants.USER_ICN, icn)
                .withClaim(VamfJwtClaimsConstants.ID_TYPE, "ICN")
                .withClaim(VamfJwtClaimsConstants.USER_SSN, ssn)
                .withClaim(VamfJwtClaimsConstants.USER_EMAIL_ADDRESS,(email==null)?"":email)
                .withClaim(VamfJwtClaimsConstants.USER_SOURCE, "MHV")
                .withClaim(VamfJwtClaimsConstants.AUTHENTICATION_AUTHORITY, jwtAuthenticationAuthority)
                .withClaim(VamfJwtClaimsConstants.USER_GENDER, jwtGender)
                .withClaim(VamfJwtClaimsConstants.USER_BIRTH_DATE, dob)
                .withClaim(VamfJwtClaimsConstants.MHV_SESSION_ID, mhvSessionId)
                .withArrayClaim(VamfJwtClaimsConstants.AUTH_ROLES_CLAIM,new String[]{VamfJwtClaimsConstants.AUTH_ROLE_VETERAN})
                .withJWTId(UUID.randomUUID().toString())
                .withSubject(icn) //  was mhvId
                .withIssuer(issuerId)
                .withExpiresAt(expireAfter.getTime())
                .withNotBefore(notBefore.getTime())
                .sign(signingAlgorithm);
		
		if(LOG.isDebugEnabled()) {
			LOG.debug(String.format("MHV JWT Generated Token:\n%s", prettyParseJwt(token)));
		}
		
		return token;
	}
	
	public boolean isTokenInvalid(String userVamfJsonWebToken) {
		
		DecodedJWT jwt = JWT.decode(userVamfJsonWebToken);
		Long notBefore = jwt.getNotBefore().getTime();
		Long expiration = jwt.getExpiresAt().getTime();
		Long now = System.currentTimeMillis();
	
		return now < notBefore || expiration < now; 
	}
	
	private String prettyParseJwt(String jwt){
		Gson gson = new GsonBuilder().setPrettyPrinting().create();
        return gson.toJson(getJsonElement(jwt));
    }
	
	private JsonElement getJsonElement(String encodedJwt) {
        List<String> jwtSections = Arrays.asList(encodedJwt.split("\\."));
        if(jwtSections.size() != 3) throw new IllegalArgumentException("Did find the standard 3-section JWT encoded string.");
        String payload = new String(Base64.decodeBase64(jwtSections.get(1).getBytes()));
        JsonParser parser = new JsonParser();
        return parser.parse(payload);
	}
}
